package @package@;

import java.lang.reflect.Constructor;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Enumeration;

import javax.sql.PooledConnection;

/**
 * @author Sibeni, Thomas Berger
 * 
 * Note: this file is a copy of
 * org.fao.fi.fishstat.commons/src/org/fao/fi/fishstat/data/common/api/ResourceManager.java
 * 
 * v3.00.0  09-Jun-2016  added closeDriver() method
 * v3.04.4  14-Mar-2018  added deRegisterDrivers() to fix JDBC memory leak
 */
public class ResourceManager {
	public static final String ORACLE_JDBC_DRIVER = "oracle.jdbc.driver.OracleDriver";
	public static final String DERBY_JDBC_EMBEDDED_DRIVER = "org.apache.derby.jdbc.EmbeddedDriver";
	public static final String POSTGREE_JDBC_DRIVER = "org.postgresql.Driver";
	
	public static final String ORACLE_POOLED_CONNECTION = "oracle.jdbc.pool.OraclePooledConnection";

	public static final String DEFAULT_JDBC_DRIVER   = DERBY_JDBC_EMBEDDED_DRIVER;
	public static final String DEFAULT_JDBC_USER     = "fishstat";
	public static final String DEFAULT_JDBC_PASSWORD = null;
	public static final String DEFAULT_JDBC_DAO_JAR_URL = null;
	public static final String DEFAULT_JDBC_SCHEMA = "FISHSTAT";

	public static final String JDBC_URL_PROPERTY = "fishstat.jdbc.url";
	public static final String JDBC_DRIVER_PROPERTY = "fishstat.jdbc.driver";
	public static final String JDBC_USER_PROPERTY = "fishstat.jdbc.user";
	public static final String JDBC_PASSWORD_PROPERTY = "fishstat.jdbc.password";
	public static final String JDBC_DAO_JAR_URL_PROPERTY = "fishstat.jdbc.dao.jar.url";
	public static final String JDBC_SCHEMA_PROPERTY = "fishstat.jdbc.schema";

	private static Driver driver = null;
	private static PooledConnection connectionsPool = null;

	private ResourceManager() {
		throw new IllegalStateException("Utility class");
	}
	 
	/**
	 * @return getProperty("fishstat.jdbc.url")
	 */
	public static synchronized String getJdbcUrl() {
		return System.getProperty(JDBC_URL_PROPERTY);
	}

	/**
	 * @param url setProperty("fishstat.jdbc.url")
	 */
	public static synchronized void setJdbcUrl(String url) {
		System.setProperty(JDBC_URL_PROPERTY, url);
	}

	/**
	 * @return getProperty("fishstat.jdbc.driver", DEFAULT_JDBC_DRIVER="org.apache.derby.jdbc.EmbeddedDriver")
	 */
	public static synchronized String getJdbcDriver() {
		return System.getProperty(JDBC_DRIVER_PROPERTY, DEFAULT_JDBC_DRIVER);
	}

	/**
	 * @param driverClassName setProperty("fishstat.jdbc.driver")
	 */
	public static synchronized void setJdbcDriver(String driverClassName) {
		System.setProperty(JDBC_DRIVER_PROPERTY, driverClassName);
	}

	/**
	 * @return getProperty("fishstat.jdbc.user", "fishstat")
	 */
	public static synchronized String getJdbcUser() {
		return System.getProperty(JDBC_USER_PROPERTY, DEFAULT_JDBC_USER);
	}

	/**
	 * @param user setProperty("fishstat.jdbc.user")
	 */
	public static synchronized void setJdbcUser(String user) {
		System.setProperty(JDBC_USER_PROPERTY, user);
	}

	/**
	 * @return getProperty("fishstat.jdbc.password", DEFAULT_JDBC_PASSWORD=null(not set))
	 */
	public static synchronized String getJdbcPassword() {
		return System.getProperty(JDBC_PASSWORD_PROPERTY, DEFAULT_JDBC_PASSWORD);
	}

	/**
	 * @param password setProperty("fishstat.jdbc.password")
	 */
	public static synchronized void setJdbcPassword(String password) {
		System.setProperty(JDBC_PASSWORD_PROPERTY, password);
	}
	
	/**
	 * @return getProperty("fishstat.jdbc.dao.jar.url", DEFAULT_JDBC_DAO_JAR_URL=null(not set))
	 */
	public static String getJdbcDaoJarUrl() {
		return System.getProperty(JDBC_DAO_JAR_URL_PROPERTY, DEFAULT_JDBC_DAO_JAR_URL);
	}

	/**
	 * @param jar setProperty("fishstat.jdbc.dao.jar.url")
	 */
	public static void setJdbcDaoJarUrl(String jar) {
		System.setProperty(JDBC_DAO_JAR_URL_PROPERTY, jar);
	}

	/**
	 * @return getProperty("fishstat.jdbc.schema", "FISHSTAT")
	 */
	public static String getJdbcSchema() {
		return System.getProperty(JDBC_SCHEMA_PROPERTY, DEFAULT_JDBC_SCHEMA);
	}

	/**
	 * @param schema setProperty("fishstat.jdbc.schema") 
	 */
	public static void setJdbcSchema(String schema) {
		System.setProperty(JDBC_SCHEMA_PROPERTY, schema);
	}

	/**
	 * shut down the whole derby system
	 * and deregister all JDBC drivers to prevent memory leaks
	 */
	public static void closeDriver() {

		if (getJdbcDriver().equals(DERBY_JDBC_EMBEDDED_DRIVER)) {
			try {
				DriverManager.getConnection("jdbc:derby:;shutdown=true");
			} catch (SQLException e) {
				System.out.println("Derby database shutdown executed OK.");
			}
		}
		
        deRegisterDrivers();
	}
	
	/**
	 * manually deregisters JDBC drivers, which prevents memory leaks to this class
	 */
	private static void deRegisterDrivers() {
        Enumeration<Driver> drivers = DriverManager.getDrivers();
        while (drivers.hasMoreElements()) {
            Driver driver = drivers.nextElement();
            System.out.println("deregistering jdbc driver: "+driver);
            try {
                DriverManager.deregisterDriver(driver);
            } catch (SQLException e) {
                System.out.println("ERROR: deregistering jdbc driver: "+driver);
                e.printStackTrace();
            }
        }
	}

	/**
	 * get the user-specified connection or get a connection from the ResourceManager
	 * @return a Connection
	 * @throws SQLException
	 */
	public static synchronized Connection getConnection() throws SQLException {
		Connection connection = null;
		if (driver == null) {
			try {
				ClassLoader loader = Thread.currentThread().getContextClassLoader();
				System.out.println("ResourceManager.embedded: loading JDBC driver: " + getJdbcDriver() + " url: " + getJdbcDaoJarUrl());
				Class<?> jdbcDriverClass = loader.loadClass( getJdbcDriver() );
				driver = (Driver) jdbcDriverClass.newInstance();
				DriverManager.registerDriver( driver );
				if (getJdbcDriver().equals(ORACLE_JDBC_DRIVER)){
					// build up an Oracle connection pool
					Constructor<?> c = Class.forName(ORACLE_POOLED_CONNECTION).getConstructor(String.class, String.class, String.class);
					connectionsPool = (PooledConnection)c.newInstance(getJdbcUrl(),getJdbcUser(),getJdbcPassword());
				}
			} catch (Exception e) {
				System.out.println( "Failed to initialise JDBC driver" );
				e.printStackTrace();
			}
		}
		
		if (getJdbcDriver().equals(ORACLE_JDBC_DRIVER)) {
			connection = connectionsPool.getConnection();
		} else {
			connection = DriverManager.getConnection(getJdbcUrl(), getJdbcUser(), getJdbcPassword()); 
		}
		return connection;
	}
	
	public static void close(Connection conn) {
		try {
			if (!getJdbcDriver().equals(DERBY_JDBC_EMBEDDED_DRIVER)) {
				if (conn != null) conn.close();
			}
		} catch (SQLException sqle)	{
			sqle.printStackTrace();
		}
	}

	public static void close(PreparedStatement stmt) {
		try {
			if (stmt != null) stmt.close();
		} catch (SQLException sqle)	{
			sqle.printStackTrace();
		}
	}

	public static void close(ResultSet rs) {
		try {
			if (rs != null) rs.close();
		} catch (SQLException sqle) {
			sqle.printStackTrace();
		}
	}
}
